import { SceneJson, ComponentObject, VariantObject, Radii } from "./type"

type Shape = {
  x: number;
  y: number;
  width: number;
  height: number;
  radii: Radii;
  color: string
}

class Scene {
  paths: Array<Path2D> = []
  shapes: Array<Shape> = []
  sceneConfig = { background: 'transparent' }
  
  constructor(json: SceneJson) {
    this.sceneConfig.background = json.a
    this.shapes = this.objectsToShapes(json.objects)
  }

  roundRect({ x, y, width, height, radii }: Shape): Path2D {
    const roundRect = new Path2D()
    roundRect.roundRect(x, y, width, height, radii)
    roundRect.closePath()
    return roundRect
  }

  /**
   * matches the first (highest) path
   */
  isInShape(ctx: CanvasRenderingContext2D, { cx, cy }) {
    const pathLen = this.paths.length
    for(let i = 0; i < pathLen; i++) {
      if(ctx.isPointInPath(this.paths[i], cx, cy)) return { match: true, pathIndex: i }
    }
    return { match: false }
  }

  objectsToShapes(objects: Array<ComponentObject | VariantObject>): Array<Shape> {
    const objLen = objects.length
    const shapes: Array<Shape> = []

    // rewire object format to shape format
    const objectToShape = ({ a, b, c, d, e, f, g, h, k }: ComponentObject): Shape => 
      ({ x: a, y: b, width: c, height: d, radii: [ e, f, g, h ], color: k })

    for (let index = 0; index < objLen; index ++) {
      switch(objects[index].t) {
        // it's an instance
        case 1:
          const variantObject = objects[index] as VariantObject
          const referenceComponent = objects.find((o) => (o.i === variantObject.r)) as ComponentObject | undefined
          if (!referenceComponent)
            throw new Error('Missing reference for this instance')
          const { c, d, e, f, g, h, t , k } = referenceComponent
          const { a, b, i } = variantObject
          shapes.push(objectToShape({ a, b, c, d, e, f, g, h, i , t, k }))
          break;
        // it's a native component
        case 0:
          const object = objects[index] as ComponentObject
          shapes.push(objectToShape(object))
          break;
        default:
          throw new Error('object.t cannot be else than 0 or 1');
      }
    }
    return shapes
  }

  drawScene(ctx: CanvasRenderingContext2D) {
    ctx.fillStyle = this.sceneConfig.background
    ctx.fillRect(0, 0, ctx.canvas.width, ctx.canvas.height)

    this.paths = []
    this.shapes.forEach((shape) => {
      const { color } = shape
      ctx.fillStyle = color
      const path = this.roundRect(shape)
      this.paths.push(path)
      const roundRect = path
      ctx.fill(roundRect)
    })
  }
}

export default Scene